<?php


class Device
{

    protected $client;

    public function __construct($client)
    {
        $this->client = $client;
    }

    public function getDeviceListSimple(){
        $result = $this->client->request(json_encode([
            'request' => 1,
            'content' => ''
        ]));
        $result = json_decode($result, true);
        return $result['content']['devices'];
    }
    public function getDeviceList()
    {
        $result = $this->client->request(json_encode([
            'request' => 1,
            'content' => ''
        ]));
        try {
            $result = json_decode($result, true);
            if (!isset($result['content']['devices'])) {
                error('32 数据获取失败');
            }
            $devices = $result['content']['devices'];
            
            //获取设备的状态  output: use array parameter
            $infoQuery = array_map(function ($device) {
                if ($device['port_num'] > 32)
                {
                    return [
                        'addr' => $device['addr'],
                        'on_off' => true,
                        'input' => true,
                        'temp_hum' => true,
                        'output' => num_bin2dec(32),
                        'output_high' => num_bin2dec($device['port_num'] - 32),
                        'phase' => isset($GLOBALS['Flag']['is3PH'])?($GLOBALS['Flag']['is3PH']?true:false):false,
                        'portCnt' => $device['port_num'],
                    ];
                }else{
                    return [
                        'addr' => $device['addr'],
                        'on_off' => true,
                        'input' => true,
                        'temp_hum' => true,
                        'output' => num_bin2dec($device['port_num']),
                        'phase' => isset($GLOBALS['Flag']['is3PH'])?($GLOBALS['Flag']['is3PH']?true:false):false,
                        'portCnt' => $device['port_num'],
                    ];
                }
                
            }, $devices);

            $infoResult = $this->client->request(json_encode([
                'request' => 3,
                'content' => [
                    'query' => $infoQuery
                ]
            ]));
            
            $infoResult = json_decode($infoResult, true);
            if (!isset($infoResult['content']['data'])) {
                error('69 数据获取失败');
            }
            $infoList = $infoResult['content']['data'];

            //组装设备的每个端口的通电状态
            $cnt = 0;
            $infoList = array_map(function ($item) use($devices,&$cnt) {
                if (isset($item['output']))
                {
                    if (count($item['output']) > 32)
                    {
                        $onOff = dec2bin_arr($item['on_off']); 
                        $onOffHigh = dec2bin_arr($item['on_off_high']); 
                        $onOff = array_merge( $onOff,$onOffHigh);
                    }else
                    {
                        $onOff = dec2bin_arr($item['on_off']);      //将设备的通电状态转化为二进制字符串数组，匹配每个端口的状态
                    }
                    $item['output'] = array_map(function ($port) use ($onOff, $item) {
                        $port['name'] = getPortName($item['addr'], $port['port']);
                        $port['status'] = (int)Arr::get($onOff, $port['port']);
                        return $port;
                    }, $item['output']);
                }else
                {
                    for ($i = 0; $i < $devices[$cnt]['port_num']; $i++) {
                        $item['output'][] = ['port' => $i,'voltage'=>0,'energy'=>0,'status'=>0,'name'=>getPortName($item['addr'], $i)];
                    }
                }
                $item['name'] = getPduName($item['addr']);
                $item['uid'] = $devices[$cnt]['uid'];
                if ($devices[$cnt]['uid'][1] == '0' || $devices[$cnt]['uid'][1] == '1')
                {
                    $item['modules'] = $devices[$cnt]['modules'];
                }
                $cnt++;
                return $item;
            }, $infoList);
            return $infoList;
        } catch (\Exception $exception) {
            error('94 数据获取失败');
        }

    }

    /**
     * 操作端口
     * @param $addr     integer  主机地址
     * @param $ports    array    端口列表
     * @param $action   integer  操作类型
     * @return bool|false|string
     */
    public function operatePort($addr, $ports, $action)
    {
        $result = $this->client->request(json_encode([
            'request' => 2,
            'content' => [
                'addr' => intval($addr),
                'ports' => array_map(function ($item) {
                    return intval($item);
                }, $ports),
                'action' => intval($action),
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    /**
     * 设置系统时间
     * @param $timestamp
     * @return false|string
     */
    public function setDateTime($timestamp)
    {

        $result = $this->client->request(json_encode([
            'request' => 12,
            'content' => [
                'time' => [
                    'year' => (int)date('Y', $timestamp),
                    'month' => (int)date('m', $timestamp),
                    'day' => (int)date('d', $timestamp),
                    'hour' => (int)date('H', $timestamp),
                    'minute' => (int)date('i', $timestamp),
                    'second' => (int)date('s', $timestamp),
                ]
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    /**
     * 电表清零
     * @param $addr
     * @return false|string
     */
    public function reset($addr)
    {
        $result = $this->client->request(json_encode([
            'request' => 13,
            'content' => [
                'addr' => $addr
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }


    /**
     * PDU 输入阈值告警
     * @param $addr
     * @param $current_max
     * @param $current_min
     * @param $voltage_max
     * @param $voltage_min
     * @param $off_current_max
     * @param $off_current_min
     * @return false|string
     */
    public function updateInputThreshold($addr, $current_max, $current_min, $voltage_max, $voltage_min, $off_current_max, $off_current_min)
    {
        $threshold = [];
        if (!is_null($current_max)) {
            $threshold['warn_upper_current'] = round((double)$current_max, 2);
        }
        if (!is_null($current_min)) {
            $threshold['warn_lower_current'] = round((double)$current_min, 2);
        }
        if (!is_null($voltage_max)) {
            $threshold['warn_upper_voltage'] = round((double)$voltage_max, 2);
        }
        if (!is_null($voltage_min)) {
            $threshold['warn_lower_voltage'] = round((double)$voltage_min, 2);
        }
        if (!is_null($off_current_max)) {
            $threshold['off_upper_current'] = round((double)$off_current_max, 2);
        }
        if (!is_null($off_current_min)) {
            $threshold['off_lower_current'] = round((double)$off_current_min, 2);
        }
        $result = $this->client->request(json_encode([
            'request' => 5,
            'content' => [
                'addr' => $addr,
                'threshold' => $threshold
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    /**
     * 输出阈值更新
     * @param $addr
     * @param $data
     * @return false|string
     */
    public function updateOutThreshold($addr, $data)
    {
        if (!$GLOBALS['Flag']['isExternPort4'])
        {
            $result = $this->client->request(json_encode([
                'request' => 4,
                'content' => [
                    'addr' => $addr,
                    'threshold' => array_map(function ($item) {
                        $dt['port'] = $item['port'];
                        isset($item['current_max']) && $dt['warn_upper_current'] = round((double)$item['current_max'], 2);
                        isset($item['current_min']) && $dt['warn_lower_current'] = round((double)$item['current_min'], 2);
                        isset($item['outage_max']) && $dt['off_upper_current'] = round((double)$item['outage_max'], 2);
                        isset($item['outage_min']) && $dt['off_lower_current'] = round((double)$item['outage_min'], 2);
                        return $dt;
                    }, $data)
                ]
            ]));
            try {
                $result = json_encode($result, true);
                return $result;
            } catch (\Exception $exception) {
                error('数据获取失败');
            }
        } else
        {
            $result = $this->client->request(json_encode([
                'request' => 4,
                'content' => [
                    'addr' => $addr,
                    'threshold' => array_map(function ($item) {
                        $dt['port'] = $item['port'];
                        isset($item['current_max']) && $dt['warn_upper_current'] = round((double)$item['current_max'], 2);
                        isset($item['current_min']) && $dt['warn_lower_current'] = round((double)$item['current_min'], 2);
                        return $dt;
                    }, $data)
                ]
            ]));
            try {
                $result = json_encode($result, true);
                return $result;
            } catch (\Exception $exception) {
                error('数据获取失败');
            }
        }

    }


    /**
     * 温湿度告警
     * @param $addr
     * @param $temperature_max
     * @param $temperature_min
     * @param $humidity_max
     * @param $humidity_min
     * @return false|string
     */
    public function updateHumitureThreshold($addr, $temperature_max, $temperature_min, $humidity_max, $humidity_min)
    {
        $result = $this->client->request(json_encode([
            'request' => 15,
            'content' => [
                'addr' => $addr,
                'threshold' => [
                    'warn_upper_temp' => round((double)$temperature_max, 2),
                    'warn_lower_temp' => round((double)$temperature_min, 2),
                    'warn_upper_humi' => round((double)$humidity_max, 2),
                    'warn_lower_humi' => round((double)$humidity_min, 2),
                ]
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }


    /**
     * 设置为主机
     * @param $addr
     * @param $slaves
     * @return false|string
     */
    // public function setMaster($addr, $slaves)
    // {
    //     $result = $this->client->request(json_encode([
    //         'request' => 10,
    //         'content' => [
    //             'role' => 1,
    //             'addr' => $addr,
    //             'slaves' => (array)$slaves
    //         ]
    //     ]));
    //     try {
    //         $result = json_encode($result, true);
    //         return $result;
    //     } catch (\Exception $exception) {
    //         error('数据获取失败');
    //     }
    // }

    /**
     * 设置为从机
     * @param $addr
     * @return false|string
     */
    // public function setSlave($addr)
    // {
    //     $result = $this->client->request(json_encode([
    //         'request' => 10,
    //         'content' => [
    //             'role' => 0,
    //             'addr' => $addr,
    //         ]
    //     ]));
    //     try {
    //         $result = json_encode($result, true);
    //         return $result;
    //     } catch (\Exception $exception) {
    //         error('数据获取失败');
    //     }
    // }


    /**
     * 计划任务管理
     * @param $id
     * @param $addr
     * @return false|string
     */
    public function createTask($id, $addr)
    {
        return $this->requestTask($id, $addr, 1);
    }

    public function updateTask($id, $addr)
    {
        return $this->requestTask($id, $addr, 2);
    }

    public function deleteTask($id, $addr)
    {
        return $this->requestTask($id, $addr, 3);
    }

    protected function requestTask($id, $addr, $action)
    {
        $result = $this->client->request(json_encode([
            'request' => 6,
            'content' => [
                'id' => $id,
                'addr' => $addr,
                'action' => $action
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    /**
     * 设置端口延迟
     * @param $addr
     * @param $data
     * @return false|string
     */
    public function setPortDelay($addr, $data)
    {
        $result = $this->client->request(json_encode([
            'request' => 11,
            'content' => [
                'addr' => $addr,
                'delay' => array_map(function ($item) {
                    $dt['port'] = $item['port'];
                    isset($item['on']) && $dt['on'] = (int)$item['on'];
                    isset($item['off']) && $dt['off'] = (int)$item['off'];
                    isset($item['restart']) && $dt['restart'] = (int)$item['restart'];
                    return $dt;
                }, $data)
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    public function setPortDelayPoweroff($addr, $data)
    {
        $result = $this->client->request(json_encode([
            'request' => 42,
            'content' => [
                'addr' => $addr,
                'set' => array_map(function ($item) {
                    $dt['port'] = $item['port'];
                    isset($item['poweroff_delay']) && $dt['off_delay'] = (int)$item['poweroff_delay'];
                    isset($item['poweroff_action']) && $dt['regain_policy'] = (int)$item['poweroff_action'];
                    isset($item['poweroff_enable']) && $dt['enabled'] = (int)$item['poweroff_enable'];
                    return $dt;
                }, $data)
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    /**
     * 重启PDU
     * @param $addr
     * @return false|string
     */
    public function restart($addr)
    {
        $result = $this->client->request(json_encode([
            'request' => 14,
            'content' => [
                'addr' => $addr,
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }


    public function sendEmailTest($smtp_server, $smtp_port, $user, $password, $receiver, $content, $addr = 0)
    {
        $result = $this->client->request(json_encode([
            'request' => 16,
            'content' => [
                'addr' => $addr,
                'smtp_server' => $smtp_server,
                'smtp_port' => $smtp_port,
                'user' => $user,
                'password' => $password,
                'receiver' => $receiver,
                'content' => $content
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    public function setPing($addr, $data)
    {

        $result = $this->client->request(json_encode([
            'request' => 17,
            'content' => [
                'addr' => $addr,
                'set' => $data
            ]
        ]));
        try {
            $result = json_encode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    public function getPing($addr,$ports = [])
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 38,
                "content" => [
                    "addr" => $addr,
                    'ports'=>array_map('intval',$ports)
                ]
            ]));
            return json_decode($result, true)['content'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }

    public function testPing($addr, $port, $ip)
    {
        $result = $this->client->request(json_encode([
            'request' => 20,
            'content' => [
                'addr' => $addr,
                'port' => $port,
                'ip' => $ip
            ]
        ]));
        try {
            $result = json_decode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败' . $result);
        }
    }


    public function testPingStatus($addr, $port)
    {
        $result = $this->client->request(json_encode([
            'request' => 21,
            'content' => [
                'addr' => $addr,
                'port' => $port,
            ]
        ]));
        try {
            $result = json_decode($result, true);
            return $result;
        } catch (\Exception $exception) {
            error('数据获取失败' . $result);
        }
    }

    public function saveLinking($addr,  $sensor_no, $controller_no, $enabled)
    {
        $this->client->request(json_encode([
            'request' => 18,
            'content' => [
                'addr' => $addr ?? 0,
                'sensor_no' => $sensor_no,
                'controller_no' => $controller_no,
                'enabled' => $enabled
            ]
        ]));
    }
    public function updateLinking($addr, $id, $sensor_no, $controller_no, $enabled)
    {
        $this->client->request(json_encode([
            'request' => 33,
            'content' => [
                'addr' => $addr ?? 0,
                'id' => $id,
                'sensor_no' => $sensor_no,
                'controller_no' => $controller_no,
                'enabled' => $enabled
            ]
        ]));
    }

    public function deleteLinking($addr, $id)
    {
        $this->client->request(json_encode([
            'request' => 19,
            'content' => [
                'addr' => $addr ?? 0,
                'ids' => [$id],
            ]
        ]));
    }

    /**
     * 查询计划任务
     * @param $addr
     * @param $start
     * @param $end
     * @return array
     */

    public function queryTasks($addr, $start = '', $end = '')
    {
        $data = [
            'addr' => intval($addr)
        ];
        if ($start) {
            $data['start_date'] = datetime2arr(local2utc($start), true);
        }
        if ($end) {
            $data['end_date'] = datetime2arr(local2utc($end), true);;
        }


        try {
            $result = $this->client->request(json_encode([
                'request' => 26,
                'content' => $data
            ]));
            $result = json_decode($result, true);
            $data = [];
            foreach ($result['content']['tasks'] as $task) {
                $task['addr'] = $result['content']['addr'];
                $task['date_time'] = utc2local("{$task['time']['year']}-{$task['time']['month']}-{$task['time']['day']}  {$task['time']['hour']}:{$task['time']['minute']}:{$task['time']['second']}");
                $utcWeekday = date('w', strtotime("{$task['time']['year']}-{$task['time']['month']}-{$task['time']['day']}  {$task['time']['hour']}:{$task['time']['minute']}:{$task['time']['second']}")); //原始的星期
                $localWeek = date('w', strtotime($task['date_time'])); //UTC的星期
                $weeDiff = $localWeek - $utcWeekday;
                $repeat = dec2bin2arr($task['repeat']);
                //把每个星期加上差值
                array_walk($repeat, function (&$item) use ($weeDiff) {
                    $item += $weeDiff;
                    if ($item > 6) {
                        $item = 0;
                    } else if ($item < 0) {
                        $item = 6;
                    }
                });
                $task['repeat'] = $repeat;
                $task['date'] = explode(' ', $task['date_time'])[0];
                $task['time'] = explode(' ', $task['date_time'])[1];
                $data[] = $task;
            }
            return $data;

        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }


    public function addTask($addr, $ports = [], $action = 0, $datetime = '', $repeat = [], $remark = '')
    {
        $originWeekday = date('w', strtotime($datetime)); //原始的星期
        $datetime = local2utc($datetime);

        $utcWeekday = date('w', strtotime($datetime)); //转化之后的utc星期
        $weeDiff = $utcWeekday - $originWeekday;

        //把每个星期加上差值
        array_walk($repeat, function (&$item) use ($weeDiff) {
            $item += $weeDiff;
            if ($item > 6) {
                $item = 0;
            } else if ($item < 0) {
                $item = 6;
            }
        });

        $data = [
            "request" => 27,
            'content' => [
                'addr' => $addr,
                'ports' => $ports,
                'action' => $action,
                'time' => datetime2arr($datetime, true),
                'repeat' => arr2bin2dec($repeat),
                'remark' => $remark,
            ]
        ];
        try {
            $result = $this->client->request(json_encode($data));
            return json_decode($result, true);
        } catch (\Exception $exception) {
            error('添加计划任务失败');
        }
    }

    public function editTask($id, $addr, $ports = [], $action = 0, $datetime = '', $repeat = [], $remark = '')
    {
        $originWeekday = date('w', strtotime($datetime)); //原始的星期
        $datetime = local2utc($datetime);

        $utcWeekday = date('w', strtotime($datetime)); //转化之后的utc星期
        $weeDiff = $utcWeekday - $originWeekday;

        //把每个星期加上差值
        array_walk($repeat, function (&$item) use ($weeDiff) {
            $item += $weeDiff;
            if ($item > 6) {
                $item = 0;
            } else if ($item < 0) {
                $item = 6;
            }
        });

        $data = [
            "request" => 28,
            'content' => [
                'id' => $id,
                'addr' => $addr,
                'ports' => $ports,
                'action' => $action,
                'time' => datetime2arr($datetime, true),
                'repeat' => arr2bin2dec($repeat),
                'remark' => $remark,
            ]
        ];

        try {
            $result = $this->client->request(json_encode($data));
            return json_decode($result, true);
        } catch (\Exception $exception) {
            error('添加计划任务失败');
        }
    }

    public function delTask($id, $addr)
    {
        $data = [
            "request" => 29,
            'content' => [
                'ids' => (array)$id,
                'addr' => $addr,
            ]
        ];
        try {
            $result = $this->client->request(json_encode($data));
            return json_decode($result, true);
        } catch (\Exception $exception) {
            error('删除计划任务失败');
        }
    }

    public function getInputThresholds($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 23,
                "content" => [
                    "addr" => $addr
                ]
            ]));
            $result = json_decode($result, true);
            if ($GLOBALS['Flag']['isExternPort4'])
            {
                return [
                    'addr' => $addr,
                    'current_max' => $result['content']['threshold']['warn_upper_current'],
                    'current_min' => $result['content']['threshold']['warn_lower_current'],
                    'voltage_max' => $result['content']['threshold']['warn_upper_voltage'],
                    'voltage_min' => $result['content']['threshold']['warn_lower_voltage'],
                    'off_current_max' => $result['content']['threshold']['off_upper_current'],
                    'off_current_min' => $result['content']['threshold']['off_lower_current'],
                ];
            }else
            {
                return [
                    'addr' => $addr,
                    'current_max' => $result['content']['threshold']['warn_upper_current'],
                    'current_min' => $result['content']['threshold']['warn_lower_current'],
                    'voltage_max' => $result['content']['threshold']['warn_upper_voltage'],
                    'voltage_min' => $result['content']['threshold']['warn_lower_voltage'],
                ];
            }
            
        } catch (\Exception $exception) {
            error('数据获取失败');
        }

    }

    public function getOutputThresholds($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 22,
                "content" => [
                    "addr" => $addr
                ]
            ]));
            $data = [];
            $result = json_decode($result, true);
            if ($GLOBALS['Flag']['isExternPort4'])
            {
                foreach ($result['content']['threshold'] as $item) {
                    $data[] = [
                        'addr' => $result['content']['addr'],
                        'current_max' => $item['warn_upper_current'],
                        'current_min' => $item['warn_lower_current'],
                        'port' => $item['port'],
                    ];
                }
            }else
            {
                foreach ($result['content']['threshold'] as $item) {
                    $data[] = [
                        'addr' => $result['content']['addr'],
                        'current_max' => $item['warn_upper_current'],
                        'current_min' => $item['warn_lower_current'],
                        'port' => $item['port'],
                        'outage_max' => $item['off_upper_current'],
                        'outage_min' => $item['off_lower_current'],
                    ];
                }
            }
            return $data;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }


    public function getHumitureThresholds($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 25,
                "content" => [
                    "addr" => $addr
                ]
            ]));
            $result = json_decode($result, true);
            return [
                'addr' => $addr,
                'temperature_max' => $result['content']['threshold']['warn_upper_temp'],
                'temperature_min' => $result['content']['threshold']['warn_lower_temp'],
                'humidity_max' => $result['content']['threshold']['warn_upper_humi'],
                'humidity_min' => $result['content']['threshold']['warn_lower_humi']
            ];
        } catch (\Exception $exception) {
            error('数据获取失败');
        }

    }

    public function getPortDelay($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 24,
                "content" => [
                    "addr" => $addr
                ]
            ]));
            $data = [];
            $result = json_decode($result, true);
            foreach ($result['content']['delay'] as $item) {
                $item['addr'] = $result['content']['addr'];
                $data[] = $item;
            }
            return $data;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    public function getPortDelayPoweroff($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 41,
                "content" => [
                    "addr" => $addr
                ]
            ]));
            $data = [];
            $result = json_decode($result, true);
            foreach ($result['content']['sets'] as $item) {
                $item['addr'] = $result['content']['addr'];
                $data[] = $item;
            }
            return $data;
        } catch (\Exception $exception) {
            error('数据获取失败');
        }
    }

    public function getPortName($addr, $ports = [])
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 31,
                "content" => [
                    "addr" => $addr,
                    "device" => true,
                    "ports" => (array)$ports
                ]
            ]));
            return json_decode($result, true)['content'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }

    public function setPortName($addr, $name, $port_names)
    {
        try {
            return $this->client->request(json_encode([
                "request" => 30,
                "content" => [
                    "addr" => $addr,
                    "device_name" => $name,
                    "port_names" => $port_names
                ]
            ]));

        } catch (\Exception $exception) {
            error('设置失败');
        }
    }


    public function getController($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 36,
                "content" => [
                    "addr" => $addr,
                ]
            ]));
            return json_decode($result, true)['content'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }

    public function setController($addr,$data)
    {
        try {
            return $this->client->request(json_encode([
                "request" => 37,
                "content" => [
                    'addr' => $addr,
                    'controllers' => $data
                ]
            ]));

        } catch (\Exception $exception) {
            error('设置失败');
        }
    }


    public function setSensor($addr, $sensors)
    {
        try {
            return $this->client->request(json_encode([
                "request" => 35,
                "content" => [
                    'addr' => $addr,
                    'sensors' => $sensors
                ]
            ]));

        } catch (\Exception $exception) {
            error('设置失败');
        }
    }

    public function getSensor($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 34,
                "content" => [
                    "addr" => $addr,
                ]
            ]));
            return json_decode($result, true)['content'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }


    public function getLinks($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 32,
                "content" => [
                    "addr" => $addr,
                ]
            ]));
            return json_decode($result, true)['content'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }

    public function getNtp($addr)
    {
        try {
            $result = $this->client->request(json_encode([
                "request" => 39,
                "content" => [
                    "addr" => $addr,
                ]
            ]));
            return json_decode($result, true)['content'] ?? ['master_ntp_server'=>'192.168.0.100','slave_ntp_server'=>'192.168.0.101'];
        } catch (\Exception $exception) {
            error('获取失败');
        }
    }

    public function setNtp($addr, $first, $second)
    {
        try {
            return $this->client->request(json_encode([
                "request" => 40,
                "content" => [
                    'addr' => $addr,
                    'master_ntp_server' => $first,
                    'slave_ntp_server' => $second
                ]
            ]));

        } catch (\Exception $exception) {
            error('设置失败');
        }
    }


}




